/* Copyright (c) 2009, 2010, Oracle and/or its affiliates. 
All rights reserved. */

/*
   NAME
      enumservices.c - Enumerate Windows NT Services

   DESCRIPTION
      Native code to be called from a JNI interface to enumerate NT Services
      which are tying up the input list of modules (dll's and exe's)

   EXPORT FUNCTION(S)
      Java_oracle_install_library_processes_ssgetRunningProcessClassW32_ssgetRunningProcessW32 - Java Interface


   INTERNAL FUNCTION(S)
      EnumServices         - Enumerates services associated with given modules
      GetAssocProcesses    - gets list of running processes assoc. w/ modules
      IsProcessAssociated  - determines whether process is assoc. w/ module
      GetServiceNamesFromProcesses - gets service names assoc. w/ processes
      ThrowNativeException - throws native exception back to JavaVM

   STATIC FUNCTION(S)
      None


   NOTES
      This code is designed to be built as a Win32 DLL which is loaded from
      Java through JNI.


   MODIFIED   (MM/DD/YYYY)
      dschrein   07/12/10 - update for 11.2
   svaggu	   16/01/2009 - Added SetPrivileges function to fix the bug7697175.	
   seramasw    06/13/07   - added winsock2 for win.X64
   mmckerle    09/23/1999 - created
   avale       10/06/1999 - Modified original exe to work as DLL callable from Java
   mmckerle    01/11/2000 - Modified to use runtime dynamic linking for psapi code
   mmckerle    06/12/2000 - Fixed problems with user-defined error codes
   mmckerle    06/12/2000 - Cleaned up memory allocation code in
                            GetServicesFromProcesses() function
*/
#include<winsock2.h>
#include <stdio.h>
#include <windows.h>
#include <winsvc.h>
#include "enumservices.h"


/*-----------------------------------------------------------------
                        PRIVATE TYPES AND CONSTANTS
 -----------------------------------------------------------------*/
// Error codes for the internal functions, which have been designed
// to coexist with system error codes.
/* -------------------------------------------------------------- */
// We use the following conventions:
//    1. Severity (31-30)
//       a. 00 = Success
//       b. 01 = Informational
//       c. 10 = Warning
//       d. 11 = Error
//    2. We set the 29th bit to 1 so there is no overlap with system codes
//    3. Reserved (28)
//    4. Facility code (27-16) (FACILITY_STORAGE = 3, FACILITY_WIN32 = 7, FACILITY_NULL = 0)
//    5. Exception code (15-0)


// Informational
#define ERROR_NO_ASSOCIATIONS 0x60000000
#define ERROR_NO_SERVICES_FOR_PROCESSES 0x60000001

// Warnings
#define ERROR_PSAPI_LIB_NOT_FOUND 0xA0070001
#define ERROR_PSAPI_FUNC_NOT_FOUND 0xA0070002

// Errors
#define ERROR_CANNOT_ENUMERATE_PROCESSES 0xE0070003
#define ERROR_CANNOT_OPEN_PROCESS 0xE0070004
#define ERROR_CANNOT_ENUMERATE_SERVICES 0xE0070005
#define ERROR_CANNOT_QUERY_SERVICE 0xE0070006
#define ERROR_CANNOT_ENUMERATE_MODULES 0xE0070007
#define ERROR_CANNOT_GET_MODULE_FILENAME 0xE0070008
#define ERROR_MEM_RESERVE_FAILURE 0xE0030001
#define ERROR_MEM_COMMIT_FAILURE 0xE0030002
#define ERROR_NOT_ENOUGH_MEMORY_RESERVED 0xE0030003
#define ERROR_MEM_FREE_FAILURE 0xE0030004
/* -------------------------------------------------------------- */

#define PAGES 10        // Initial number of pages to reserve for each buffer
#define MAX_MODULES 200 // Maximum number of modules program can handle
#define MAX_SVCNAME 128 // Maximum length of a service name in chars


// Typedef declarations for the PSAPI functions
typedef BOOL (WINAPI *ENUMPROCESSES)(DWORD *, DWORD, DWORD *);
typedef BOOL (WINAPI *ENUMPROCESSMODULES)(HANDLE, HMODULE*, DWORD, LPDWORD);
typedef DWORD (WINAPI *GETMODULEFILENAMEEX)(HANDLE, HMODULE, LPTSTR, DWORD);

// Needed to account for two different function names for GetModuleFileNameEx
#ifdef UNICODE
#define GETMODFILENAME "GetModuleFileNameExW"
#else
#define GETMODFILENAME "GetModuleFileNameExA"
#endif // !UNICODE


// Stores location of extracted psapi dll
TCHAR szPsapiLocation[MAX_PATH];


/*-----------------------------------------------------------------
                        STATIC FUNCTION DECLARATIONS
 -----------------------------------------------------------------*/
DWORD EnumServices(CONST TCHAR[][MAX_PATH], DWORD, TCHAR[][MAX_SVCNAME]);
DWORD GetAssocProcesses(CONST TCHAR[][MAX_PATH], DWORD, TCHAR[][MAX_PATH]);
BOOL IsProcessAssociated(DWORD, CONST TCHAR[][MAX_PATH], DWORD, TCHAR[],
                         ENUMPROCESSMODULES, GETMODULEFILENAMEEX);
DWORD GetServicesFromProcesses(CONST TCHAR[][MAX_PATH], DWORD,
                               TCHAR[][MAX_SVCNAME]);
void ThrowNativeException(JNIEnv *, char *);
void SetPrivileges();


/*-----------------------------------------------------------------
                        EXPORT FUNCTIONS
 -----------------------------------------------------------------*/

/*---------------------- ssGetRunningProcessW32 ------------------*/
/*
   NAME
      ssGetRunningProcessW32- Java Interface to EnumServices

   DESCRIPTION
      This is the Java Interface to the other native internal functions

   REQUIRES


   PARAMETERS
      Formals:
         env - pointer to a JNI struct which contains Java functions
         obj - number of modules in above array
         filenames - array of input modules (DLL or EXE) to process

   RETURNS
      This return values for ssgetRunningProcessW32() are:
         NULL if failure (a fatal exception was thrown)
         NULL if No Services were found (no exception was thrown)
         An array of running services associated with the input modules (DLL or EXE)

   EXCEPTIONS
      This function throws Native Java Exceptions with the following values:
       - RuntimeException if one of the following occurs:
            More than 200 input modules were specified
            EnumerateProcesses() failed
            OpenProcess() failed
            EnumProcessModules() failed
            GetModuleFilenameEx() failed
            Reservation of memory failed for an unknown reason
            Commital of memory failed for an unknown reason
            More than PLIMIT pages of memory was required to read in the services
            EnumerateServices() failed
            QueryServiceConfig() failed
            An unknown error occured
       - NoServicesForProcessException if some of the input modules were running,
            but no services were associated with them (most likely a running app)

   EXAMPLES

   NOTES
      See comments
*/
JNIEXPORT jobjectArray JNICALL Java_oracle_install_library_processes_ssgetRunningProcessClassW32_ssgetRunningProcessW32
     (JNIEnv *env ,jobject obj, jobjectArray filenames, jstring jpsapiLocation)

{
   TCHAR aServiceNames[MAX_MODULES][MAX_SVCNAME];
   TCHAR aModules[MAX_MODULES][MAX_PATH];
   DWORD dwServices;
   DWORD dwModsToSearch = 0;
   int i;
   unsigned int j;
   char *exString;

   jclass stringClass;
   jint jListSize;
   jboolean *isCopy = 0;
   jstring retVal = (*env)->NewStringUTF(env, "");
   jobjectArray retArray;


   // Check the size of the input list of modules to ensure it is not too large
   jListSize = (jint) (*env)->GetArrayLength(env,filenames);
   if (jListSize <= 0)
   {
      return NULL;
   }
   else if(jListSize > MAX_MODULES)
   {
      exString = "RuntimeException";
      ThrowNativeException(env,exString);
      return NULL;
   }

   // Transfer this input list of modules into the array of TCHARS.
   for (i = 0; i < jListSize; i++)
   {
      lstrcpy(aModules[i],(TCHAR *)(* env)->GetStringUTFChars(env,(jstring)(*env)->GetObjectArrayElement(env,filenames,i), isCopy));
   }

   // Store the DLL location in szPsapiLocation
   if(!lstrcpy(szPsapiLocation, (TCHAR *)(*env)->GetStringUTFChars(env,jpsapiLocation,isCopy)))
   {
      exString = "RuntimeException";
      ThrowNativeException(env, exString);
      return NULL;
   }


   // Do the work...
   dwModsToSearch = jListSize;
   if((dwServices = EnumServices(aModules, dwModsToSearch, aServiceNames)))
   {
      stringClass = (*env)->GetObjectClass(env,retVal);
      retArray = (*env)->NewObjectArray(env,dwServices,stringClass,retVal);
      for(j = 0; j < dwServices; j++)
      {
         retVal = (*env)->NewStringUTF(env,aServiceNames[j]);
         (*env)->SetObjectArrayElement(env,retArray,j,retVal);
      }
      return retArray;
   }
   else
   {
      // Exception handling:
      // If there were no associated processes for the input list of modules,
      // or if there was a problem in loading the psapi DLL, we simply return
      // the NULL list rather than throwing a native exception.
      switch(GetLastError())
      {
      case ERROR_NO_ASSOCIATIONS:
         retArray = NULL;
         return retArray;
      case ERROR_PSAPI_LIB_NOT_FOUND:
         retArray = NULL;
         return retArray;
      case ERROR_PSAPI_FUNC_NOT_FOUND:
         retArray = NULL;
         return retArray;
      case ERROR_CANNOT_ENUMERATE_PROCESSES:
         exString = "RuntimeException";
         ThrowNativeException(env,exString);
         return NULL;
      case ERROR_CANNOT_OPEN_PROCESS:
         exString = "RuntimeException";
         ThrowNativeException(env,exString);
         return NULL;
      case ERROR_CANNOT_ENUMERATE_MODULES:
         exString = "RuntimeException";
         ThrowNativeException(env,exString);
         return NULL;
      case ERROR_CANNOT_GET_MODULE_FILENAME:
         exString = "RuntimeException";
         ThrowNativeException(env,exString);
         return NULL;
      case ERROR_MEM_RESERVE_FAILURE:
         exString = "RuntimeException";
         ThrowNativeException(env,exString);
         return NULL;
      case ERROR_MEM_COMMIT_FAILURE:
         exString = "RuntimeException";
         ThrowNativeException(env,exString);
         return NULL;
      case ERROR_NOT_ENOUGH_MEMORY_RESERVED:
         exString = "RuntimeException";
         ThrowNativeException(env,exString);
         return NULL;
      case ERROR_CANNOT_ENUMERATE_SERVICES:
         exString = "RuntimeException";
         ThrowNativeException(env,exString);
         return NULL;
      case ERROR_CANNOT_QUERY_SERVICE:
         exString = "RuntimeException";
         ThrowNativeException(env,exString);
         return NULL;
      case ERROR_NO_SERVICES_FOR_PROCESSES:
         exString = "NoServicesForProcessException";
         ThrowNativeException(env,exString);
         return NULL;
      default:
         exString = "RuntimeException";
         ThrowNativeException(env,exString);
         return NULL;
      }
   }
}


/*-----------------------------------------------------------------
                        INTERNAL FUNCTIONS
 -----------------------------------------------------------------*/
/*--------------------------- EnumServices -----------------------*/
/*
   NAME
      EnumServices - Enumerate NT Services associated with module

   DESCRIPTION
      Enumerates NT Services associated with given modules (DLL or EXE)

   REQUIRES


   PARAMETERS
      Formals:
         aModulesToSearchFor - constant array of modules to enum services for
         dwModsToSearch      - number of modules in above array
         aAssociatedServices - array of services associated with modules

   RETURNS
      The return codes returned by enumservices():
         0 if failure OR if there were no (running) services associated w/ mods
         <number of services returned> if success

   EXCEPTIONS
      None

   EXAMPLES

   NOTES
      See comments
*/
DWORD EnumServices(CONST TCHAR aModulesToSearchFor[][MAX_PATH],
                   DWORD dwModsToSearch,
                   TCHAR aAssociatedServices[][MAX_SVCNAME])
{
   TCHAR aProcessesInUse[MAX_MODULES][MAX_PATH];
   DWORD dwProcesses;
   DWORD dwServices;
   SetPrivileges();

   if((dwProcesses = GetAssocProcesses(aModulesToSearchFor, dwModsToSearch,
                                       aProcessesInUse)))
      if((dwServices = GetServicesFromProcesses(aProcessesInUse, dwProcesses,
                                                aAssociatedServices)))
         return dwServices;

   return 0;
}


/*--------------------------- GetAssocProcesses -----------------------*/
/*
   NAME
      GetAssocProcesses - gets list of running processes associated w/ modules

   DESCRIPTION
      Gets a list of running processes associated with the input list of
      modules (DLL or EXE)

   REQUIRES
     The following funtions are required from the PSAPI:
         EnumProcesses()
         EnumProcessModules()
         GetModuleFileNameEx()
      The runtime version of these functions are in the psapi DLL.  The definitions
      are in psapi.h and the link library psapi.lib


   PARAMETERS
      Formals:
         aModulesToSearchFor - constant array of modules to search for
         dwModsToSearch - number of modules to search for
         aProcessesInUse - subset of above modules which are in use

   RETURNS
      The return codes returned by GetAssocProcesses():
         0 if failure OR if no (running) processes were found using mods
         <number of processes returned> if success

   EXCEPTIONS
      This function calls SetLastError() with the following values:
         - ERROR_CANNOT_ENUMERATE_PROCESSES if EnumProcesses() failed
         - ERROR_NO_ASSOCIATIONS if no processes were using specified modules
         - ERROR_PSAPI_LIB_NOT_FOUND if the psapi DLL was not found on the system
         - ERROR_PSAPI_FUNC_NOT_FOUND if a PSAPI function was not found in the DLL

   EXAMPLES

   NOTES
      See comments
*/
DWORD GetAssocProcesses(CONST TCHAR aModulesToSearchFor[][MAX_PATH],
                        DWORD dwModsToSearch,
                        TCHAR aProcessesInUse[][MAX_PATH])
{
   DWORD aProcessIds[MAX_MODULES], dwNeeded, dwProcesses;
   unsigned int i;
   DWORD dwPcount = 0;           // counter for number of processes in use
   TCHAR szExeName[MAX_PATH];

   // Used for runtime loading of psapi DLL
   HINSTANCE hinstPsapi = NULL;                       // handle to the loaded instance for psapi.dll
   ENUMPROCESSES myEnumProcesses = NULL;              // pointer to EnumProcesses()
   ENUMPROCESSMODULES myEnumProcessModules = NULL;    // pointer to EnumProcessModules()
   GETMODULEFILENAMEEX myGetModuleFileNameEx = NULL;  // pointer to GetModuleFileNameEx()



   // First we need to load the psapi library (the dll extension is implicit)
   hinstPsapi = LoadLibrary(szPsapiLocation);

   // If the library was not found, we exit gracefully
   if(hinstPsapi == NULL)
   {
      SetLastError(ERROR_PSAPI_LIB_NOT_FOUND);
      return 0;
   }

   // The library was found.
   // Now we need to retreive the function pointer to EnumProcesses()
   myEnumProcesses = (ENUMPROCESSES) GetProcAddress(hinstPsapi, "EnumProcesses");

   // Get pointer to EnumProcessModules()
   myEnumProcessModules = (ENUMPROCESSMODULES) GetProcAddress(hinstPsapi,
                                                              "EnumProcessModules");

   // Get pointer to GetModuleFileNameEx()
   myGetModuleFileNameEx = (GETMODULEFILENAMEEX) GetProcAddress(hinstPsapi,
                                                                GETMODFILENAME);

   // If there was a problem retrieving any of the functions, exit gracefully
   if((myEnumProcesses == NULL) ||
      (myEnumProcessModules == NULL) ||
      (myGetModuleFileNameEx == NULL))
   {
      SetLastError(ERROR_PSAPI_FUNC_NOT_FOUND);
      FreeLibrary(hinstPsapi);
      return 0;
   }

   // Get the list of process identifiers.
   if(!(*myEnumProcesses)(aProcessIds, sizeof(aProcessIds), &dwNeeded))
   {
      SetLastError(ERROR_CANNOT_ENUMERATE_PROCESSES);
      return 0;
   }

   // Calculate how many process identifiers were returned.
   dwProcesses = dwNeeded / sizeof(DWORD);

   // For each Process ID, determine whether the modules for that process
   // match the modules in the input module name list.
   // If yes, then add them to the modulesInUse array
   for ( i = 0; i < dwProcesses; i++ )
   {
      if(IsProcessAssociated(aProcessIds[i], aModulesToSearchFor,
                             dwModsToSearch,
                             szExeName,
                             myEnumProcessModules,
                             myGetModuleFileNameEx))
         lstrcpy(aProcessesInUse[dwPcount++],szExeName);

      else
      {
         // We need to see if there was an error if the process was not
         // associated.  We return immediately if the reason was a failure
         // in on of the API's.
         switch(GetLastError())
         {
         case ERROR_NO_ASSOCIATIONS:  // we continue in this case
            break;
         case ERROR_CANNOT_OPEN_PROCESS:
            FreeLibrary(hinstPsapi);
            return 0;
         case ERROR_CANNOT_ENUMERATE_MODULES:
            FreeLibrary(hinstPsapi);
            return 0;
         case ERROR_CANNOT_GET_MODULE_FILENAME:
            FreeLibrary(hinstPsapi);
            return 0;
         default:
            break;
         }
      }
   }

   // If there were no fatal errors reported, then we simply did not find any
   // of the modules specified to be in use (because they were not associated
   // with any running processes.  We indicate as such and return.
   if(dwPcount == 0)
      SetLastError(ERROR_NO_ASSOCIATIONS);

   FreeLibrary(hinstPsapi);
   return dwPcount;
}


/*--------------------------- IsProcessAssociated -----------------------*/
/*
   NAME
      IsProcessAsociated - gets list of running processes associated w/ modules

   DESCRIPTION
      Gets a list of running processes associated with the input list of
      modules (DLL or EXE)

   REQUIRES


   PARAMETERS
      Formals:
         dwProcessID - the process ID of process in question
         aModulesToSearchFor - constant array of modules to search for
         dwModsToSearch - number of modules in above array
         lpExeName - the name of the process assoc. with process ID

   RETURNS
      The return codes returned by IsProcessAssociated():
         FALSE if failure
         TRUE if success

   EXCEPTIONS
      This function calls SetLastError() with the following values:
         - ERROR_NO_ASSOCIATIONS if process is not associated w/ any mods
         - ERROR_CANNOT_GET_MODULE_FILENAME if GetModuleFilenameEx() failed
         - ERROR_CANNOT_ENUMERATE_MODULES if EnumProcessModules() failed uncharacteristically
         - ERROR_CANNOT_OPEN_PROCESS if OpenProcess() failed uncharacteristically

   EXAMPLES

   NOTES
      See comments
*/
BOOL IsProcessAssociated(DWORD dwProcessID,
                         CONST TCHAR aModulesToSearchFor[][MAX_PATH],
                         DWORD dwModsToSearch,
                         TCHAR lpExeName[],
                         ENUMPROCESSMODULES myEnumProcessModules,
                         GETMODULEFILENAMEEX myGetModuleFileNameEx)
{

   HMODULE hModuleHandles[MAX_MODULES];
   HANDLE hProcess;
   DWORD dwNeeded;
   DWORD dwModules;
   LPTSTR lpExtension;
   unsigned int i, j;


   // First get a handle for the process in question.  We do not allow
   // this process handle to be inherited by any other process created by
   // the current one
   hProcess = OpenProcess(PROCESS_QUERY_INFORMATION | PROCESS_VM_READ,
                             FALSE, dwProcessID);


   // OpenProcess() will fail on the any process that gives an access violation
   // when user-level code tries to access them.  Common processes with this
   // restriction include the Idle process and the CSRSS process.  Thus we
   // check hProcess before we call EnumProcessModules
   if(hProcess)
   {
      if((*myEnumProcessModules)(hProcess, hModuleHandles, sizeof(hModuleHandles),
                            &dwNeeded))
      {
         // Get the number of modules for the current process
         dwModules = dwNeeded / sizeof(HMODULE);

         for(i = 0; i < dwModules; i++)
         {
            TCHAR szModuleName[MAX_PATH];

            // Get the full path to the module's file.
            if((*myGetModuleFileNameEx)(hProcess, hModuleHandles[i], szModuleName,
                                   sizeof(szModuleName)))
            {
               // If this is the first module, store the exename for later use
               // We set this to NULL if there was no match or if an error occured
               if(i == 0)
               {
                  // We only want to copy the Exename upto the .exe extension for
                  // comparison.  This is because some of the service ImagePath's
                  // might leave off the .exe extension.
                  lpExtension = &(szModuleName[lstrlen(szModuleName) - lstrlen(".exe")]);
                  if(!lstrcmpi(lpExtension, ".exe"))
                     lstrcpyn(lpExeName, szModuleName,
                              lstrlen(szModuleName) - lstrlen("exe"));
                  else
                     lstrcpy(lpExeName, szModuleName);
               }

               // Now determine whether this module name is relevant (i.e.,whether
               // it is in the initial list of module names to look for)
               // If it is, we immediately break out of the loop and return TRUE.
               // This way we prevent double entries in case a two input modules
               // are associated with the same exe.
               for(j = 0; j < dwModsToSearch; j++)
                  if(!lstrcmpi(szModuleName, aModulesToSearchFor[j]))
                     return TRUE;
            }
            else
            {
               SetLastError(ERROR_CANNOT_GET_MODULE_FILENAME);
               lpExeName = NULL;
               return FALSE;
            }
         }

        // If we get here, we have successfully processed all the module
        // names for the current process, and we have not found a match
        // among the module names that we are searching for.
        SetLastError(ERROR_NO_ASSOCIATIONS);
        lpExeName = NULL;
        return FALSE;
      }
      else
      {
         lpExeName = NULL;
         switch(GetLastError())
         {
         case ERROR_NOACCESS:                   // occurs for the System Process
            SetLastError(ERROR_NO_ASSOCIATIONS);
            return FALSE;
         case ERROR_PARTIAL_COPY:               // occurs for the System Process in W2K
            SetLastError(ERROR_NO_ASSOCIATIONS);
            return FALSE;
         default:                               // generic EnumProcessModules() error
            SetLastError(ERROR_CANNOT_ENUMERATE_MODULES);
            return FALSE;
         }
      }
   }
   else
   {
      lpExeName = NULL;
      switch(GetLastError())
      {
      case ERROR_INVALID_PARAMETER:          // occurs for the Idle Process (fake process)
         SetLastError(ERROR_NO_ASSOCIATIONS);
         return FALSE;
      case ERROR_ACCESS_DENIED:              // occurs for any process which has denied user-level access
         SetLastError(ERROR_NO_ASSOCIATIONS);
         return FALSE;
      default:                               // generic OpenProcess() error
         SetLastError(ERROR_CANNOT_OPEN_PROCESS);
         return FALSE;
      }
   }
}


/*------------------------ GetServicesFromProcesses -----------------------*/
/*
   NAME
      GetServicesFromProcesses -  returns service names assoc. w/ processes

   DESCRIPTION
      Returns a list of service names associated with the input list of
      process names

   REQUIRES


   PARAMETERS
      Formals:
         aExeNames     - the list of exenames
         dwNumExes     - the number of exe's in the list
         aServiceNames - the list of associated Services

   RETURNS
      The return codes returned by GetServicesFromProcesses():
         0 if failure OR if no services were assoc. with set of processes
         <number of services in list> if success

   EXCEPTIONS
      This function calls SetLastError() with the following values:
         - ERROR_MEM_RESERVE_FAILURE if the memory reservation fails
         - ERROR_MEM_COMMIT_FAILURE if memory commital fails
         - ERROR_NOT_ENOUGH_MEMORY_RESERVED if not enough memory was reserved
         - ERROR_CANNOT_ENUMERATE_SERVICES if EnumServiceStatus() failed
         - ERROR_CANNOT_QUERY_SERVICE if QueryServiceConfig() failed

   EXAMPLES

   NOTES
      See comments
*/
DWORD GetServicesFromProcesses(CONST TCHAR aExeNames[][MAX_PATH], DWORD dwNumExes,
                               TCHAR aServiceNames[][MAX_SVCNAME])
{
   unsigned int i, j;                     // Generic loop variables
   DWORD dwSvcCount = 0;                  // Counter for the aServiceNames array
   SC_HANDLE hScm;                        // Handle to the service control manager
   DWORD dwServbufSize = 0;               // Size of the buffer for service information
   DWORD dwSpaceNeeded;                   // Additional space needed in buffer if not enough
   DWORD dwNumServices;                   // Number of services returned
   DWORD dwResumeLocation;                // Place to resume if not enough buffer space
   SC_HANDLE hService;                    // Handle to service
   DWORD dwConfigbufSize = 0;             // Size of service config buffer
   DWORD dwCfgBytesNeeded;                // Bytes needed if buffer too small

   LPENUM_SERVICE_STATUS lpServbuf;       // Pointer to buffer for service information
   LPQUERY_SERVICE_CONFIG lpConfigbuf;    // Pointer to buffer for service config info
   LPENUM_SERVICE_STATUS lpNxtServPage;   // Address of next svc info buffer page to commit
   LPQUERY_SERVICE_CONFIG lpNxtCfgPage;   // Address of next svc conf info buffer page to commit

   SYSTEM_INFO sSysInfo;                  // Useful information about the system
   DWORD dwPageSize;                      // Size of the system page
   LPVOID lpvServbufBase;                 // Base address of our service info buffer
   LPVOID lpvConfigbufBase;               // Base address of our service config info buffer
   LPVOID lpvResult;                      // Generic pointer used as result of VirtualAlloc()
   BOOL bSuccess;                         // Stores Success/Failure of VirtualFree() fuction

   int iCompareString;                    // Stores CompareString() return info


   // First, populate the system information structure, and store the page size
   GetSystemInfo(&sSysInfo);
   dwPageSize = sSysInfo.dwPageSize;

   // Next reserve pages for each of the buffers lpServbuf and lpConfigBuf
   lpvServbufBase = VirtualAlloc(
                                 NULL,             // system selects address
                                 PAGES*dwPageSize, // size of allocation
                                 MEM_RESERVE,      // allocate reserved pages
                                 PAGE_READWRITE);  // protection = no access

   if(lpvServbufBase == NULL)  // the initial reservation of memory failed
   {
      // error processing code goes here
      SetLastError(ERROR_MEM_RESERVE_FAILURE);
      return 0;
   }
   lpServbuf = lpNxtServPage = (LPENUM_SERVICE_STATUS) lpvServbufBase;


   // Get a handle to the Service Control Manager so that we can get
   // service information for this machine.
   hScm = NULL;
   hScm = OpenSCManager(NULL, SERVICES_ACTIVE_DATABASE, SC_MANAGER_ENUMERATE_SERVICE);


   // Now get a list of all non-driver services on this machine which are currently
   // "active".
   dwResumeLocation = 0;
   if(!EnumServicesStatus(hScm,
                          SERVICE_WIN32,
                          SERVICE_ACTIVE,
                          lpServbuf,
                          dwServbufSize,
                          &dwSpaceNeeded,
                          &dwNumServices,
                          &dwResumeLocation))
   {
      if(GetLastError() == ERROR_MORE_DATA)    // the buffer size is too small
      {
         // In this case, we can commit more pages to the buffer and call
         // the enumerate services API again.
         // Note: since pages won't have been committed (only reserved)
         //       at this point, we expect to enter this block the first
         //       time we call EnumServicesStatus() and commit based on
         //       the info it gives us.

         if(dwSpaceNeeded > (PAGES*dwPageSize))
         {
            // We need to commit more pages than we have currently reserved
            // Free original block and re-reserve
            bSuccess = VirtualFree(
                                   lpvServbufBase,  // base address of block
                                   0,               // release the entire block
                                   MEM_RELEASE);    // release the pages
            if(!bSuccess)
            {
               SetLastError(ERROR_MEM_FREE_FAILURE);
               return 0;
            }

            lpvServbufBase = VirtualAlloc(
                                          NULL,
                                          dwSpaceNeeded,
                                          MEM_RESERVE,
                                          PAGE_READWRITE);

            if(lpvServbufBase == NULL)
            {
               // error processing code goes here
               SetLastError(ERROR_MEM_RESERVE_FAILURE);
               return 0;
            }
            lpServbuf = lpNxtServPage = (LPENUM_SERVICE_STATUS) lpvServbufBase;
         }

         dwServbufSize = dwSpaceNeeded;
         lpvResult = VirtualAlloc(
                                  (LPVOID) lpNxtServPage, // base address of block
                                  dwServbufSize,          // size of buffer to commit
                                  MEM_COMMIT,             // Commit pages to physical memory
                                  PAGE_READWRITE);        // read/write access

         if(lpvResult == NULL)
         {
            // error processing goes here
            // Before returning free the memory already reserved
            bSuccess = VirtualFree(
                                   lpvServbufBase,  // base address of block
                                   0,               // release the entire block
                                   MEM_RELEASE);    // release the pages

            SetLastError(ERROR_MEM_COMMIT_FAILURE);
            return 0;
         }
         lpServbuf = (LPENUM_SERVICE_STATUS) lpvResult;


         // Now call the function again
         dwResumeLocation = dwSpaceNeeded = dwNumServices = 0;
         if(!EnumServicesStatus(hScm,
                                SERVICE_WIN32,
                                SERVICE_ACTIVE,
                                lpServbuf,
                                dwServbufSize,
                                &dwSpaceNeeded,
                                &dwNumServices,
                                &dwResumeLocation))
         {

            // Now we fail for real.
            bSuccess = VirtualFree(
                                   lpvServbufBase,
                                   0,
                                   MEM_RELEASE | MEM_DECOMMIT);

            SetLastError(ERROR_CANNOT_ENUMERATE_SERVICES);
            return 0;

         }
      }
      else
      {
         // Since no commital of pages will have been done at this point,
         // we can simply release the entire block for both buffers.
         bSuccess = VirtualFree(
                                lpvServbufBase,  // base address of block
                                0,               // release the entire block
                                MEM_RELEASE);    // release the pages

         SetLastError(ERROR_CANNOT_ENUMERATE_SERVICES);
         return 0;
      }
   }

   // At this point lpServbuf is an array of dwNumServices ENUM_SERVICE_STATUS
   // entries. Step through the array and process each service to build the
   // table of service names corresponding to aExeNames

   // First we reserve memory for a buffer to hold service config information
   lpvConfigbufBase = VirtualAlloc(
                                   NULL,             // system selects address
                                   PAGES*dwPageSize, // size of allocation
                                   MEM_RESERVE,      // allocate reserved pages
                                   PAGE_READWRITE);  // protection = no access

   if(lpvConfigbufBase == NULL)  // the initial reservation of memory failed
   {
      // error processing code goes here
      // Before returning free the memory already reserved
      bSuccess = VirtualFree(
                             lpvServbufBase,
                             0,
                             MEM_RELEASE | MEM_DECOMMIT);

      SetLastError(ERROR_MEM_RESERVE_FAILURE);
      return 0;
   }
   lpConfigbuf = lpNxtCfgPage = (LPQUERY_SERVICE_CONFIG) lpvConfigbufBase;

   for(i = 0; i < dwNumServices; i++)
   {
      // We need to get configuration for each of the services to determine
      // whether the image path matches the exe names we wish to find
      // services for.

      // To do this we first need to get a handle to the service
      hService = OpenService(hScm, lpServbuf[i].lpServiceName,
                             (SERVICE_QUERY_CONFIG | SERVICE_QUERY_STATUS));

      if(!QueryServiceConfig(hService, lpConfigbuf, dwConfigbufSize,
                             &dwCfgBytesNeeded))
      {
         // call GetLastError to see if buffer was insufficient
         if(GetLastError() == ERROR_INSUFFICIENT_BUFFER)
         {
            // In this case, we can commit more pages to the buffer and call
            // the QueryServiceConfig() API again.
            // Note: since pages won't have been committed (only reserved)
            //       at this point, we expect to enter this block the first
            //       time we call EnumServicesStatus() and commit based on
            //       the info it gives us.

            if(dwCfgBytesNeeded > PAGES*dwPageSize)
            {
               // We need to commit more pages than we have currently reserved
               // Free original block and re-reserve
               bSuccess = VirtualFree(
                                      lpvConfigbufBase,
                                      0,
                                      MEM_RELEASE);
               if(!bSuccess)
               {
                  SetLastError(ERROR_MEM_FREE_FAILURE);
                  return 0;
               }

               lpvConfigbufBase = VirtualAlloc(
                                               NULL,
                                               dwCfgBytesNeeded,
                                               MEM_RESERVE,
                                               PAGE_READWRITE);

               if(lpvConfigbufBase == NULL)
               {
                  // error processing code goes here
                  SetLastError(ERROR_MEM_RESERVE_FAILURE);
                  return 0;
               }
               lpConfigbuf = lpNxtCfgPage = (LPQUERY_SERVICE_CONFIG) lpvConfigbufBase;
            }

            dwConfigbufSize = dwCfgBytesNeeded;
            lpvResult = VirtualAlloc(
                                     (LPVOID) lpNxtCfgPage, // next page to commit
                                     dwConfigbufSize,       // size of buffer to commit
                                     MEM_COMMIT,            // allocate committed pages
                                     PAGE_READWRITE);       // read/write access
            if(lpvResult == NULL)
            {
               // error processing goes here
               // Before returning decommit/release the memory already allocated
               bSuccess = VirtualFree(
                                      lpvServbufBase,
                                      0,
                                      MEM_RELEASE | MEM_DECOMMIT);
               bSuccess = VirtualFree(
                                      lpvConfigbufBase,  // base address of block
                                      0,                 // release the entire block
                                      MEM_RELEASE);      // release the pages

               SetLastError(ERROR_MEM_COMMIT_FAILURE);
               return 0;
            }
            lpConfigbuf = (LPQUERY_SERVICE_CONFIG) lpvResult;

            // Now call the function again
            dwCfgBytesNeeded = 0;
            if(!QueryServiceConfig(hService, lpConfigbuf, dwConfigbufSize,
                                   &dwCfgBytesNeeded))
            {
               // Now we fail for real.
               bSuccess = VirtualFree(
                                      lpvServbufBase,
                                      0,
                                      MEM_RELEASE | MEM_DECOMMIT);

               bSuccess = VirtualFree(
                                      lpvConfigbufBase,
                                      0,
                                      MEM_RELEASE | MEM_DECOMMIT);

               SetLastError(ERROR_CANNOT_QUERY_SERVICE);
               return 0;
            }
         }
         else
         {
            // Since no commital of pages will have been done at this point,
            // we can simply release the entire block for both buffers.
            bSuccess = VirtualFree(
                                   lpvServbufBase,
                                   0,
                                   MEM_RELEASE | MEM_DECOMMIT);
            bSuccess = VirtualFree(
                                   lpvConfigbufBase,  // base address of block
                                   0,                 // release the entire block
                                   MEM_RELEASE);      // release the pages

            SetLastError(ERROR_CANNOT_QUERY_SERVICE);
            return 0;
         }
      }

      // At this point, we have successfully retrieved the Configuration
      // information for the service.  Now we can get its image path and
      // check to see if it matches any of the exes that are running.
      for(j = 0; j < dwNumExes; j++)
      {
         // We compare the BinaryPathName and the Exename only up to the
         // length of the Exename.  This is because the BinaryPathName
         // may include some command line arguments after the exename
         iCompareString = CompareString(LOCALE_USER_DEFAULT,
                                        NORM_IGNORECASE,
                                        aExeNames[j],
                                        -1,
                                        lpConfigbuf->lpBinaryPathName,
                                        lstrlen(aExeNames[j]));
         switch(iCompareString)
         {
         case CSTR_EQUAL:  // We have a match!
            lstrcpy(aServiceNames[dwSvcCount++], lpConfigbuf->lpDisplayName);
            break;
         default:
            break;
         }

      }

      // Now we can free the memory for the ConfigBuf (no longer needed)
      bSuccess = VirtualFree(
                             lpConfigbuf,     // base address of block
                             dwConfigbufSize,  // bytes of committed pages
                             MEM_DECOMMIT);    // decommit the pages

      // Zero out lpConfigbuf and dwConfigbufSize for use next time
      lpConfigbuf = NULL;
      dwConfigbufSize = 0;

      // Close the service handle to the currently open service
      CloseServiceHandle(hService);
   }

   // Finally, we need to do some cleaning up:
   // 1. Close the service handle for the service control manager
   CloseServiceHandle(hScm);

   // 2. Release the memory buffer associated with lpConfigbuf
   bSuccess = VirtualFree(
                         lpvConfigbufBase, // base address of block
                         0,                // release the entire block
                         MEM_RELEASE);     // release the pages

   // 3. Decommit and release the memory associated with lpServbuf
   bSuccess = VirtualFree(
                          lpvServbufBase,
                          0,
                          MEM_RELEASE | MEM_DECOMMIT);

   // If there were no services associated with any of the processes specified
   // then we indicate as such by setting the error.  This means that some of
   // the modules that were in the initial list were in use, and that we found
   // the processes associated with these modules, but that none of the running
   // services were responsible for tying up these modules.  The process which
   // is tying up the module must be an application of some sore.
   if(dwSvcCount == 0)
      SetLastError(ERROR_NO_SERVICES_FOR_PROCESSES);

   return dwSvcCount;
}

/*------------------------ ThrowNativeException -----------------------*/
/*
   NAME
      ThrowNativeException -  throws an exception back to Java

   DESCRIPTION
      Throws an exception back to Java through the JNI Interface

   REQUIRES


   PARAMETERS
      Formals:
         env      - pointer to a JNI struct which contains Java functions
         exString - the name of the exception that is to be thrown

   RETURNS
      None

   EXCEPTIONS
      None

   EXAMPLES

   NOTES
      See comments
*/
void ThrowNativeException(JNIEnv *env , char* exString)
{
  jmethodID mid;
  jclass actException;
  jobject exception;
  jstring err;

  err = (*env)->NewStringUTF(env, exString);
  actException = (*env)->FindClass(env, "oracle/sysman/oii/oiil/OiilNativeException");
  mid = (*env)->GetMethodID(env, actException, "<init>", "(Ljava/lang/String;)V");
  exception = (*env)->NewObject(env, actException, mid, err);
  (*env)->Throw(env, exception);
}

/* 

Function Description:

	Set Debug  privileges 
	We have added this function to fix the bug 7697175. 
	The bug  is because of session 0 isolation in case of Vista and Longhorn.
    Now on Vista and Longhorn no user can log in to session 0. And all services
    and system processes will be running in session 0. Due to session 0 isolation
    when we will try to open the process handle it will not be allowed and so we
    are not able to detect this. We need to use new API and rewrite running
    process query for Vista and Longhorn to fix out the issue.  So to handle out this issue to our current dll we have to give sufficient priveleges. This we are giving in current funtion.

Arguments:

Returns: void
*/

void SetPrivileges()
{
	HANDLE hToken=NULL;
	LUID	PrivNameValue;


	TOKEN_PRIVILEGES Privileges;
	DWORD dwRet;

	if (OpenProcessToken(
			GetCurrentProcess(),
			TOKEN_ADJUST_PRIVILEGES | TOKEN_QUERY,
			&hToken))
	{

		if (LookupPrivilegeValue(NULL,SE_DEBUG_NAME,&PrivNameValue)) {
			Privileges.PrivilegeCount=1;
			Privileges.Privileges[0].Luid=PrivNameValue;
			Privileges.Privileges[0].Attributes=SE_PRIVILEGE_ENABLED;
			if (!AdjustTokenPrivileges(hToken,FALSE,&Privileges,sizeof(Privileges),NULL,&dwRet) || 
				GetLastError() == ERROR_NOT_ALL_ASSIGNED) {
				//ShowError(NULL,"Could not set %s privilege some option's may not work",SE_DEBUG_NAME);
			}
		}
	
		CloseHandle(hToken);
	}
}


